using System;
using System.Xml;
using System.Collections.Generic;
using System.Text;

namespace XMLShapeLib
{
    public class XMLShaper
    {
        /// <summary>
        /// Gets a formatted XML string in the specified format 
        /// </summary>
        /// <param name="XMLString"></param>
        /// <param name="format"></param>
        /// <returns></returns>
        public static string GetXMLAs(string XMLString, string mapName)
        {
            Mapper mapper = new Mapper(mapName); 
            // Set the source and destination
            XmlDocument sourceDoc = new XmlDocument();
            sourceDoc.LoadXml(XMLString);
            // Use a sample XML file
            XmlDocument destinationDoc = Utils.GetXMLFromURL(Environment.CurrentDirectory + "\\XMLConfig\\" + mapName + ".xml");

            switch (mapName)
            {
                case "UPS":
                    return GetXMLAsUPS(sourceDoc, destinationDoc, mapper); 
                default:
                    return string.Empty;
            }
        }

        /// <summary>
        /// Maps xml element values using a mapper
        /// </summary>
        /// <param name="sourceDoc"></param>
        /// <param name="destinationDoc"></param>
        /// <param name="mapper"></param>
        /// <returns></returns>
        private static string GetXMLAsUPS(XmlDocument sourceDoc, XmlDocument destinationDoc, Mapper mapper)
        {
            //Remove the first OpenShipment and use it as a template
            XmlNode OpenShipment = destinationDoc.GetElementsByTagName("OpenShipment")[0];
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(destinationDoc.NameTable);
            nsmgr.AddNamespace("ups", "x-schema:OpenShipments.xdr");

            //Loop over all the orders we downloaded
            XmlNodeList orderList = sourceDoc.GetElementsByTagName("Orders");
            foreach (XmlNode order in orderList)
            {
                //Create a new UPS order
                XmlNode shipment = OpenShipment.Clone();
                //Replace the tag values shipment using the mapper
                shipment = ReplaceXMLValues(shipment, order, mapper, nsmgr);
                // Put in custom values here
                LoadUPSCustomValues(order, shipment, nsmgr);
                destinationDoc.FirstChild.AppendChild(shipment);
            }
            // Remove the first OpenShipment
            destinationDoc.FirstChild.RemoveChild(OpenShipment);
            return destinationDoc.InnerXml;
        }

        private static XmlNode ReplaceXMLValues(XmlNode sourceNode, XmlNode destinationNode, Mapper mapper, XmlNamespaceManager nsmgr)
        {
            //Replace the tag values shipment using the mapper
            foreach (string key in mapper.TagDictionary.Keys)
            {
                //Look for the value in the tag and update it in the destination document
                if (sourceNode.SelectSingleNode(mapper.TagDictionary[key], nsmgr) != null && destinationNode.SelectSingleNode(key) != null)
                    sourceNode.SelectSingleNode(mapper.TagDictionary[key], nsmgr).InnerText = destinationNode.SelectSingleNode(key).InnerText;
            }

            return sourceNode;
        }

        private static void LoadUPSCustomValues(XmlNode loadFrom, XmlNode loadTo, XmlNamespaceManager nsmgr)
        {
            //Move name: ShipFirstName and ShipLastName into Attention
            loadTo.SelectSingleNode("//ups:Attention", nsmgr).InnerText = loadFrom.SelectSingleNode("ShipFirstName").InnerText + " " + loadFrom.SelectSingleNode("ShipLastName").InnerText;
            //If there is not company, load the name into CompanyOrName
            if (string.IsNullOrEmpty(loadFrom.SelectSingleNode("ShipCompanyName ").InnerText))
                loadTo.SelectSingleNode("//ups:CompanyOrName", nsmgr).InnerText = loadTo.SelectSingleNode("//ups:Attention", nsmgr).InnerText;
        }
    }
}
